"""Compute uncertainty measures after generating answers."""
from collections import defaultdict
import logging
import os
import pickle
import numpy as np
import wandb

import torch
from analyze_results import analyze_run
from uncertainty.data.data_utils import load_ds
from uncertainty.uncertainty_measures.p_ik import get_p_ik
from uncertainty.uncertainty_measures.semantic_entropy import get_semantic_ids
from uncertainty.uncertainty_measures.semantic_entropy import logsumexp_by_id
from uncertainty.uncertainty_measures.semantic_entropy import predictive_entropy
from uncertainty.uncertainty_measures.semantic_entropy import predictive_entropy_rao
from uncertainty.uncertainty_measures.semantic_entropy import cluster_assignment_entropy
from uncertainty.uncertainty_measures.semantic_entropy import context_entails_response
from uncertainty.uncertainty_measures.semantic_entropy import EntailmentDeberta
from uncertainty.uncertainty_measures.semantic_entropy import EntailmentGPT4
from uncertainty.uncertainty_measures.semantic_entropy import EntailmentGPT35
from uncertainty.uncertainty_measures.semantic_entropy import EntailmentGPT4Turbo
from uncertainty.uncertainty_measures.semantic_entropy import EntailmentLlama
from uncertainty.uncertainty_measures.envidence import EvidenceModel
from uncertainty.uncertainty_measures import p_true as p_true_utils
from uncertainty.uncertainty_measures.self_consistency import compute_self_consistency
from uncertainty.utils import utils

import sys
from modeling_selfcheck import SelfCheckLLMPrompt
device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
self_check_llm_model = "mistralai/Mistral-7B-Instruct-v0.2"
# import pdb;pdb.set_trace()
selfcheck_prompt = SelfCheckLLMPrompt(self_check_llm_model, device)

from tqdm import tqdm
utils.setup_logger()

EXP_DETAILS = 'experiment_details.pkl'


def main(args):

    if args.train_wandb_runid is None:
        args.train_wandb_runid = args.eval_wandb_runid
    # import pdb; pdb.set_trace()
    user = os.environ['USER']
    scratch_dir = os.getenv('SCRATCH_DIR', '.')
    wandb_dir = f'{scratch_dir}/{user}/uncertainty'
    slurm_jobid = os.getenv('SLURM_JOB_ID', None)
    project = "semantic_uncertainty" if not args.debug else "semantic_uncertainty_debug"
    if args.assign_new_wandb_id:
        logging.info('Assign new wandb_id.')
        api = wandb.Api()
        old_run = api.run(f'{args.restore_entity_eval}/{project}/{args.eval_wandb_runid}')
        wandb.init(
            entity=args.entity,
            project=project,
            dir=wandb_dir,
            notes=f'slurm_id: {slurm_jobid}, experiment_lot: {args.experiment_lot}',
            # For convenience, keep any 'generate_answers' configs from old run,
            # but overwrite the rest!
            # NOTE: This means any special configs affecting this script must be
            # called again when calling this script!
            config={**old_run.config, **args.__dict__},
        )

        def restore(filename):
            old_run.file(filename).download(
                replace=True, exist_ok=False, root=wandb.run.dir)

            class Restored:
                name = f'{wandb.run.dir}/{filename}'

            return Restored
    else:
        logging.info('Reuse active wandb id.')

        def restore(filename):
            class Restored:
                name = f'{wandb.run.dir}/{filename}'
            return Restored

    if args.train_wandb_runid != args.eval_wandb_runid:
        logging.info(
            "Distribution shift for p_ik. Training on embeddings from run %s but evaluating on run %s",
            args.train_wandb_runid, args.eval_wandb_runid)

        is_ood_eval = True  # pylint: disable=invalid-name
        api = wandb.Api()
        old_run_train = api.run(f'{args.restore_entity_train}/semantic_uncertainty/{args.train_wandb_runid}')
        filename = 'train_generations.pkl'
        old_run_train.file(filename).download(
            replace=True, exist_ok=False, root=wandb.run.dir)
        with open(f'{wandb.run.dir}/{filename}', "rb") as infile:
            train_generations = pickle.load(infile)
        wandb.config.update(
            {"ood_training_set": old_run_train.config['dataset']}, allow_val_change=True)
    else:
        is_ood_eval = False  # pylint: disable=invalid-name
        if args.compute_p_ik or args.compute_p_ik_answerable:
            train_generations_pickle = restore('train_generations.pkl')
            with open(train_generations_pickle.name, 'rb') as infile:
                train_generations = pickle.load(infile)
    # import pdb; pdb.set_trace()
    wandb.config.update({"is_ood_eval": is_ood_eval}, allow_val_change=True)

    # Load entailment model.
    if args.compute_predictive_entropy:
        logging.info('Beginning loading for entailment model.')
        if args.entailment_model == 'deberta':
            entailment_model = EntailmentDeberta()
        elif args.entailment_model == 'gpt-4':
            entailment_model = EntailmentGPT4(args.entailment_cache_id, args.entailment_cache_only)
        elif args.entailment_model == 'gpt-3.5':
            entailment_model = EntailmentGPT35(args.entailment_cache_id, args.entailment_cache_only)
        elif args.entailment_model == 'gpt-4-turbo':
            entailment_model = EntailmentGPT4Turbo(args.entailment_cache_id, args.entailment_cache_only)
        elif 'llama' in args.entailment_model.lower():
            entailment_model = EntailmentLlama(args.entailment_cache_id, args.entailment_cache_only, args.entailment_model)
        else:
            raise ValueError
        logging.info('Entailment model loading complete.')

    if args.compute_p_true_in_compute_stage:
        # This is usually not called.
        old_exp = restore(EXP_DETAILS)
        with open(old_exp.name, "rb") as infile:
            old_exp = pickle.load(infile)

        if args.reuse_entailment_model:
            pt_model = entailment_model.model
        else:
            pt_model = utils.init_model(old_exp['args'])

        pt_train_dataset, pt_validation_dataset = load_ds(
            old_exp['args'].dataset, add_options=old_exp['args'].use_mc_options,
            seed=args.random_seed)
        del pt_validation_dataset

        # Reduce num generations used in p_true if needed!
        if not args.use_all_generations:
            if args.use_num_generations == -1:
                raise ValueError
            num_gen = args.use_num_generations
        else:
            num_gen = args.num_generations

        p_true_few_shot_prompt, p_true_responses, len_p_true = p_true_utils.construct_few_shot_prompt(
            model=pt_model,
            dataset=pt_train_dataset,
            indices=old_exp['p_true_indices'],
            prompt=old_exp['prompt'],
            brief=old_exp['BRIEF'],
            brief_always=old_exp['args'].brief_always and old_exp['args'].enable_brief,
            make_prompt=utils.get_make_prompt(old_exp['args']),
            num_generations=num_gen,
            metric=utils.get_metric(old_exp['args'].metric))
        del p_true_responses
        wandb.config.update(
            {'p_true_num_fewshot': len_p_true}, allow_val_change=True)
        wandb.log(dict(len_p_true=len_p_true))

        logging.info('Generated few-shot prompt for p_true.')
        logging.info(80*'#')
        logging.info('p_true_few_shot_prompt: %s', p_true_few_shot_prompt)
        logging.info(80*'#')

    if args.recompute_accuracy:
        # This is usually not enabled.
        logging.warning('Recompute accuracy enabled. This does not apply to precomputed p_true!')
        metric = utils.get_metric(args.metric)
    # import pdb; pdb.set_trace()
    # Restore outputs from `generate_answrs.py` run.
    result_dict_pickle = restore('uncertainty_measures.pkl')
    with open(result_dict_pickle.name, "rb") as infile:
        result_dict = pickle.load(infile)
    result_dict['semantic_ids'] = []


    validation_generations_pickle = restore('validation_generations.pkl')
    with open(validation_generations_pickle.name, 'rb') as infile:
        validation_generations = pickle.load(infile)

    entropies = defaultdict(list) #创建一个字典，当访问不存在的键时，会自动为该键创建一个空列表
    validation_embeddings, validation_is_true, validation_answerable, validation_llm_head_inputs, validation_down_proj_inputs, responses_list = [], [], [], [], [], []
    p_trues = []
    # down_proj_inputs
    
    count = 0  # pylint: disable=invalid-name

    def is_answerable(generation):
        return len(generation['reference']['answers']['text']) > 0

    # Loop over datapoints and compute validation embeddings and entropies.
    # import pdb;pdb.set_trace()
    log_probs = []
    for idx, tid in tqdm(enumerate(validation_generations)):
        # import pdb; pdb.set_trace()
        example = validation_generations[tid]
        question = example['question']
        context = example['context']
        full_responses = example["responses"]
        most_likely_answer = example['most_likely_answer']
        multisample_answer = [x[0] for x in full_responses]
        if not args.use_all_generations:
            if args.use_num_generations == -1:
                raise ValueError
            responses = [fr[0] for fr in full_responses[:args.use_num_generations]]
        else:
            responses = [fr[0] for fr in full_responses]

        if args.recompute_accuracy:
            logging.info('Recomputing accuracy!')
            if is_answerable(example):
                acc = metric(most_likely_answer['response'], example, None)
            else:
                acc = 0.0  # pylint: disable=invalid-name
            validation_is_true.append(acc)
            logging.info('Recomputed accuracy!')

        else:
            validation_is_true.append(most_likely_answer['accuracy'])

        validation_answerable.append(is_answerable(example))
        validation_embeddings.append(most_likely_answer['embedding'])
        responses_list.append(responses)
        # down_proj_inputs
        validation_down_proj_inputs.append(most_likely_answer['down_proj_inputs'])
        validation_llm_head_inputs.append(most_likely_answer['llm_head_inputs'])
        # import pdb;pdb.set_trace()
        logging.info('validation_is_true: %f', validation_is_true[-1])
        log_probs.append(most_likely_answer["token_log_likelihoods"])
        if args.compute_predictive_entropy:
            # Token log likelihoods. Shape = (n_sample, n_tokens)
            if not args.use_all_generations:
                log_liks = [r[1] for r in full_responses[:args.use_num_generations]]
            else:
                log_liks = [r[1] for r in full_responses]

            for i in log_liks:
                assert i
            
            if args.compute_context_entails_response:
                # Compute context entails answer baseline.
                entropies['context_entails_response'].append(context_entails_response(
                    context, responses, entailment_model))
            # import pdb; pdb.set_trace()
            if args.condition_on_question and args.entailment_model == 'deberta': 
                responses = [f'{question} {r}' for r in responses] # list 包括multisample的句子(10个)，每个句子都带有问题

            # Compute semantic ids.
            # pdb-step in
            semantic_ids = get_semantic_ids(
                responses, model=entailment_model,
                strict_entailment=args.strict_entailment, example=example)

            result_dict['semantic_ids'].append(semantic_ids)

            # Compute entropy from frequencies of cluster assignments.
            entropies['cluster_assignment_entropy'].append(cluster_assignment_entropy(semantic_ids))

            # Length normalization of generation probabilities.
            log_liks_agg = [np.mean(log_lik) for log_lik in log_liks]

            # Compute naive entropy.
            entropies['regular_entropy'].append(predictive_entropy(log_liks_agg))

            # Compute semantic entropy.
            log_likelihood_per_semantic_id = logsumexp_by_id(semantic_ids, log_liks_agg, agg='sum_normalized') #每个semantic id的log p
            pe = predictive_entropy_rao(log_likelihood_per_semantic_id)
            entropies['semantic_entropy'].append(pe)
            # import pdb;pdb.set_trace()
            selfcheck_score = selfcheck_prompt.predict(
                sentences = [ most_likely_answer["response"]],                          # list of sentences
                sampled_passages = multisample_answer, # list of sampled passages
                verbose = False, # whether to show a progress bar)
            )
            entropies['self_check'].append(np.float64(selfcheck_score))
            # import pdb; pdb.set_trace()
            # pylint: disable=invalid-name
            log_str = 'semantic_ids: %s, avg_token_log_likelihoods: %s, entropies: %s'
            # entropies_fmt = ', '.join([f'{i}:{j[-1]:.2f}' for i, j in entropies.item()])
            # pylint: enable=invalid-name
            logging.info(80*'#')
            logging.info('NEW ITEM %d at id=`%s`.', idx, tid)
            logging.info('Context:')
            logging.info(example['context'])
            logging.info('Question:')
            logging.info(question)
            logging.info('True Answers:')
            logging.info(example['reference'])
            logging.info('Low Temperature Generation:')
            logging.info(most_likely_answer['response'])
            logging.info('Low Temperature Generation Accuracy:')
            logging.info(most_likely_answer['accuracy'])
            logging.info('High Temp Generation:')
            logging.info([r[0] for r in full_responses])
            logging.info('High Temp Generation:')
            # logging.info(log_str, semantic_ids, log_liks_agg, entropies_fmt)
            # logging.info(log_str, semantic_ids, log_liks_agg)

        if args.compute_p_true_in_compute_stage:
            p_true = p_true_utils.calculate_p_true(
                pt_model, question, most_likely_answer['response'],
                responses, p_true_few_shot_prompt,
                hint=old_exp['args'].p_true_hint)
            p_trues.append(p_true)
            logging.info('p_true: %s', np.exp(p_true))

        count += 1
        if count >= args.num_eval_samples:
            logging.info('Breaking out of main loop.')
            break

    logging.info('Accuracy on original task: %f', np.mean(validation_is_true))
    validation_is_false = [1.0 - is_t for is_t in validation_is_true]
    result_dict['validation_is_false'] = validation_is_false

    validation_unanswerable = [1.0 - is_a for is_a in validation_answerable]
    result_dict['validation_unanswerable'] = validation_unanswerable
    logging.info('Unanswerable prop on validation: %f', np.mean(validation_unanswerable))

    if 'uncertainty_measures' not in result_dict:
        result_dict['uncertainty_measures'] = dict()

    if args.compute_predictive_entropy:
        result_dict['uncertainty_measures'].update(entropies)
        # import pdb; pdb.set_trace()
    if args.compute_p_ik or args.compute_p_ik_answerable:
        # Assemble training data for embedding classification.
        train_is_true, train_embeddings, train_answerable = [], [], []
        for tid in train_generations:
            most_likely_answer = train_generations[tid]['most_likely_answer']
            train_embeddings.append(most_likely_answer['embedding'])
            train_is_true.append(most_likely_answer['accuracy'])
            train_answerable.append(is_answerable(train_generations[tid]))
        train_is_false = [0.0 if is_t else 1.0 for is_t in train_is_true]
        train_unanswerable = [0.0 if is_t else 1.0 for is_t in train_answerable]
        logging.info('Unanswerable prop on p_ik training: %f', np.mean(train_unanswerable))

    if args.compute_p_ik:
        logging.info('Starting training p_ik on train embeddings.')
        # Train classifier of correct/incorrect from embeddings.
        p_ik_predictions = get_p_ik(
            train_embeddings=train_embeddings, is_false=train_is_false,
            eval_embeddings=validation_embeddings, eval_is_false=validation_is_false)
        result_dict['uncertainty_measures']['p_ik'] = p_ik_predictions
        logging.info('Finished training p_ik on train embeddings.')
    # import pdb; pdb.set_trace()
    if args.compute_p_ik_answerable:
        # Train classifier of answerable/unanswerable.
        p_ik_predictions = get_p_ik(
            train_embeddings=train_embeddings, is_false=train_unanswerable, 
            eval_embeddings=validation_embeddings, eval_is_false=validation_unanswerable)
        result_dict['uncertainty_measures']['p_ik_unanswerable'] = p_ik_predictions

    if args.compute_p_true_in_compute_stage:
        result_dict['uncertainty_measures']['p_false'] = [1 - p for p in p_trues]
        result_dict['uncertainty_measures']['p_false_fixed'] = [1 - np.exp(p) for p in p_trues]
    if args.compute_self_consistency:
        result_dict['uncertainty_measures']['self_consistency'] = []
        for responses in responses_list:
            # import pdb; pdb.set_trace()
            # responses = [r[0] for r in responses]
            # responses = [r[0] for r in responses]
            result_dict['uncertainty_measures']['self_consistency'].append(compute_self_consistency(responses))
    if args.compute_probalibility_method:
        # import pdb; pdb.set_trace()
        result_dict['uncertainty_measures']['entropy'] = []
        result_dict['uncertainty_measures']['ln-entropy'] = []
        result_dict['uncertainty_measures']['prob'] = []
        result_dict['uncertainty_measures']['ln-prob'] = []
        for probs in log_probs:
            result_dict['uncertainty_measures']['entropy'].append(predictive_entropy(probs))
            result_dict['uncertainty_measures']['ln-entropy'].append(predictive_entropy(probs)/len(probs))
            result_dict['uncertainty_measures']['prob'].append(torch.sum(torch.exp(torch.tensor(probs))).item())
            result_dict['uncertainty_measures']['ln-prob'].append(torch.mean(torch.exp(torch.tensor(probs))).item())
        # import pdb; pdb.set_trace()

    if args.attention_evidence_metric:
        print('Attention evidence metric')
        state_dict = torch.load("mof-7b-0-down_proj-weight.pth",weights_only=True).to(device)
        model_bias = torch.zeros((1,state_dict.shape[1] )).to(device)
        evidence_model = EvidenceModel(state_dict.T, model_bias) #llama2chat: torch.Size([4096, 11008]) mof: torch.Size([5120, 13824])
        result_dict['uncertainty_measures']['attention_conflict'] = []
        result_dict['uncertainty_measures']['attention_ignorance'] = []
        for down_proj_input in tqdm(validation_down_proj_inputs):
            conflict_value = 0
            ig_value = 0
            for feature in down_proj_input:
                # import pdb; pdb.set_trace()
                evidence_weights = evidence_model.get_evidence_weights(feature.squeeze(0))
                # evidence_weights = evidence_model.get_evidence_weights(feature.squeeze(0))
                conflict_value += evidence_model.get_evidence_conflict()
                ig_value += evidence_model.get_evidence_ignorance()
            result_dict['uncertainty_measures']['attention_conflict'].append(conflict_value.item() / len(down_proj_input))
            result_dict['uncertainty_measures']['attention_ignorance'].append(ig_value.item() / len(down_proj_input))
            # for feature in down_proj_input[:-1]:
            #     evidence_weights = evidence_model.get_evidence_weights(feature.squeeze(0).T)
            #     conflict_value += evidence_model.get_evidence_conflict()
            #     ig_value += evidence_model.get_evidence_ignorance()
            # result_dict['uncertainty_measures']['attention_conflict'].append(conflict_value.item() )
            # result_dict['uncertainty_measures']['attention_ignorance'].append(ig_value)
        
    if args.llm_head_evidence_metric:
        # import pdb; pdb.set_trace()
        print('llmhead evidence metric')
        state_dict = torch.load("mof-llm-head-weight.pth",weights_only=True).to(device)
        model_bias = torch.zeros((1,state_dict.shape[1] )).to(device)
        evidence_model = EvidenceModel(state_dict.T, model_bias) #llama2chat: torch.Size([4096, 11008]) mof: torch.Size([5120, 13824])

        result_dict['uncertainty_measures']['llm_head_conflict'] = []
        result_dict['uncertainty_measures']['llm_head_ignorance'] = []
        for llm_head_input in tqdm(validation_llm_head_inputs):
            conflict_value = 0
            ig_value = 0
            for feature in llm_head_input:
                evidence_weights = evidence_model.get_evidence_weights(feature.squeeze(0))
                conflict_value += evidence_model.get_evidence_conflict()
                ig_value += evidence_model.get_evidence_ignorance()
            # import pdb;pdb.set_trace()
            result_dict['uncertainty_measures']['llm_head_conflict'].append(conflict_value.item() / len(llm_head_input))
            result_dict['uncertainty_measures']['llm_head_ignorance'].append(ig_value.item() / len(llm_head_input))
        # import pdb; pdb.set_trace()
    utils.save(result_dict, 'uncertainty_measures.pkl')
    logging.info('Finished computing uncertainty measures.')

    if args.compute_predictive_entropy:
        entailment_model.save_prediction_cache()

    if args.analyze_run:
        # Follow up with computation of aggregate performance metrics.
        logging.info(50 * '#X')
        logging.info('STARTING `analyze_run`!')
        analyze_run(wandb.run.id) #'vthxhcop
        logging.info(50 * '#X')
        logging.info('FINISHED `analyze_run`!')


if __name__ == '__main__':
    parser = utils.get_parser(stages=['compute'])
    args, unknown = parser.parse_known_args()  # pylint: disable=invalid-name
    if unknown:
        raise ValueError(f'Unkown args: {unknown}')

    logging.info("Args: %s", args)

    main(args)
